<?php
/**
 * Plugin Name: Astrovibe Horoscopes – Shortcode
 * Description: Ajoute un shortcode [astrovibe_horoscopes] qui affiche des liens d'horoscopes du jour (Astrovibe.fr) avec un style responsive.
 * Version: 1.0.4
 * Author: www.astrovibe.fr
 * Author URI: https://www.astrovibe.fr
 * Plugin URI: https://www.astrovibe.fr/astrologie-guide/plugin-wordpress-horoscope-gratuit/
 * License: GPLv2 or later
 * Text Domain: astrovibe-horoscopes-shortcode
 */

if (!defined('ABSPATH')) { exit; }

if (!class_exists('Astrovibe_Horoscopes_Shortcode')) {
    class Astrovibe_Horoscopes_Shortcode {
        const HANDLE = 'astrovibe-horoscopes-shortcode';

        public function __construct() {
            add_shortcode('astrovibe_horoscopes', array($this, 'render_shortcode'));
            add_action('wp_enqueue_scripts', array($this, 'maybe_enqueue_assets'));
        }

        /**
         * Charge les assets uniquement si le shortcode est présent.
         */
        public function maybe_enqueue_assets() {
            if (!is_singular()) { return; }
            global $post;
            if (!$post) { return; }
            if (has_shortcode($post->post_content, 'astrovibe_horoscopes')) {
                $this->enqueue_assets();
            }
        }

        /**
         * Enqueue styles & scripts inline (sans heredoc).
         */
        private function enqueue_assets() {
            // CSS inline
            wp_register_style(self::HANDLE, false, array(), '1.0.4');
            wp_enqueue_style(self::HANDLE);

            $css_lines = array(
                '#liens-horoscope-jour {',
                '  display: grid;',
                '  grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));',
                '  gap: 1.5rem;',
                '  margin-top: 2rem;',
                '  padding: 0 1rem;',
                '}',
                '#liens-horoscope-jour a {',
                '  display: block;',
                '  text-align: center;',
                '  background: linear-gradient(135deg, #2fe2c4 0%, #23a6d5 100%);',
                '  color: #ffffff;',
                "  font-family: 'Poppins','Segoe UI',sans-serif;",
                '  font-size: 1rem;',
                '  font-weight: 600;',
                '  text-decoration: none;',
                '  border-radius: 32px;',
                '  padding: 0.8rem 1.2rem;',
                '  box-shadow: 0 4px 10px rgba(0,0,0,0.12);',
                '  transition: transform 0.2s ease, background 0.2s ease;',
                '}',
                '#liens-horoscope-jour a:hover {',
                '  background: linear-gradient(135deg, #1dd8b4 0%, #209bd2 100%);',
                '  transform: translateY(-3px);',
                '}',
                '#credit-astro {',
                '  text-align: center;',
                '  margin-top: 1.5rem;',
                "  font-family: 'Poppins','Segoe UI',sans-serif;",
                '  font-size: 0.9rem;',
                '}',
                '#credit-astro a {',
                '  color: #23a6d5;',
                '  text-decoration: none;',
                '  font-weight: 500;',
                '}',
            );
            wp_add_inline_style(self::HANDLE, implode("\n", $css_lines));

            // JS inline
            wp_register_script(self::HANDLE, '', array(), '1.0.4', true);
            wp_enqueue_script(self::HANDLE);

            $js_lines = array(
                '(function(){',
                '  function ready(fn){',
                "    if(document.readyState !== 'loading'){ fn(); }",
                "    else { document.addEventListener('DOMContentLoaded', fn); }",
                '  }',
                '  ready(function(){',
                "    var container = document.getElementById('liens-horoscope-jour');",
                '    if(!container){ return; }',
                "    var signes = ['belier','taureau','gemeaux','cancer','lion','vierge','balance','scorpion','sagittaire','capricorne','verseau','poisson'];",
                "    var moisNoms = ['janvier','février','mars','avril','mai','juin','juillet','août','septembre','octobre','novembre','décembre'];",
                '    var today = new Date();',
                '    var jour = today.getDate();',
                '    var mois = moisNoms[today.getMonth()];',
                '    var annee = today.getFullYear();',
                '    signes.forEach(function(signe){',
                "      var url = 'https://www.astrovibe.fr/horoscope/horoscope-quotidien-' + signe + '-' + jour + '-' + mois + '-' + annee + '/';",
                "      var label = 'Horoscope ' + (signe.charAt(0).toUpperCase() + signe.slice(1)) + ' du ' + jour + ' ' + mois + ' ' + annee;",
                '      var link = document.createElement("a");',
                '      link.href = url;',
                '      link.textContent = label;',
                '      link.setAttribute("rel","noopener noreferrer");',
                '      link.setAttribute("target","_blank");',
                '      container.appendChild(link);',
                '    });',
                '    setTimeout(function(){',
                "      var credit = document.querySelector('#credit-astro a');",
                "      if (!credit || !credit.href || credit.href.indexOf('astrovibe.fr') === -1 || credit.textContent.indexOf('Horoscopes du jour') === -1) {",
                "        container.innerHTML = '';",
                "        var creditWrap = document.getElementById('credit-astro');",
                '        if (creditWrap) { creditWrap.innerHTML = ""; }',
                '      }',
                '    }, 1000);',
                '  });',
                '})();'
            );
            wp_add_inline_script(self::HANDLE, implode("\n", $js_lines));
        }

        /**
         * Rendu du shortcode.
         * Usage: [astrovibe_horoscopes class="mt-6"]
         */
        public function render_shortcode($atts = array()) {
            $atts = shortcode_atts(
                array(
                    'class' => '',
                ),
                $atts,
                'astrovibe_horoscopes'
            );

            // S'assure que les assets sont chargés même via certains builders
            $this->enqueue_assets();

            $html  = '<div id="liens-horoscope-jour" class="' . esc_attr($atts['class']) . '"></div>';
            // Crédit toujours affiché
            $html .= '<div id="credit-astro">'
                . '<a href="https://www.astrovibe.fr/horoscope/" target="_blank" rel="noopener">Horoscopes du jour</a> fournis par Astrovibe.fr'
                . '</div>';

            return $html;
        }
    }

    new Astrovibe_Horoscopes_Shortcode();
}
